<?php

namespace FASTRANSPLUGIN\Element;

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Border;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Plugin;

/**
 * Elementor button widget.
 * Elementor widget that displays a button with the ability to control every
 * aspect of the button design.
 *
 * @since 1.0.0
 */
class Our_Services_V2 extends Widget_Base {

	/**
	 * Get widget name.
	 * Retrieve button widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'fastrans_our_services_v2';
	}

	/**
	 * Get widget title.
	 * Retrieve button widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Our Services V2', 'fastrans' );
	}

	/**
	 * Get widget icon.
	 * Retrieve button widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-library-open';
	}

	/**
	 * Get widget categories.
	 * Retrieve the list of categories the button widget belongs to.
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since  2.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'fastrans' ];
	}
	
	/**
	 * Register button widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'our_services_v2',
			[
				'label' => esc_html__( 'Our Services V2', 'fastrans' ),
			]
		);
		$this->add_control(
			'subtitle',
			[
				'label'       => __( 'Sub Title', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your Sub title', 'fastrans' ),
			]
		);
		$this->add_control(
			'title',
			[
				'label'       => __( 'Title', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXTAREA,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your title', 'fastrans' ),
			]
		);
		$this->add_control(
			'query_number',
			[
				'label'   => esc_html__( 'Number of post', 'fastrans' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 3,
				'min'     => 1,
				'max'     => 100,
				'step'    => 1,
			]
		);
		$this->add_control(
			'query_orderby',
			[
				'label'   => esc_html__( 'Order By', 'fastrans' ),
				'label_block' => true,
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => array(
					'date'       => esc_html__( 'Date', 'fastrans' ),
					'title'      => esc_html__( 'Title', 'fastrans' ),
					'menu_order' => esc_html__( 'Menu Order', 'fastrans' ),
					'rand'       => esc_html__( 'Random', 'fastrans' ),
				),
			]
		);
		$this->add_control(
			'query_order',
			[
				'label'   => esc_html__( 'Order', 'fastrans' ),
				'label_block' => true,
				'type'    => Controls_Manager::SELECT,
				'default' => 'DESC',
				'options' => array(
					'DESC' => esc_html__( 'DESC', 'fastrans' ),
					'ASC'  => esc_html__( 'ASC', 'fastrans' ),
				),
			]
		);
		$this->add_control(
            'query_category', 
			[
			  'type' => Controls_Manager::SELECT,
			  'label' => esc_html__('Category', 'fastrans'),
			  'label_block' => true,
			  'options' => get_service_categories()
			]
		);
		$this->add_control(
            'show_pagination',
			[
				'label' => __( 'Enable/Disable Pagination', 'fastrans' ),
				'type'     => Controls_Manager::SWITCHER,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enable/Disable Pagination', 'fastrans' ),
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'section_style_1',
			[
				'label' => __( 'Section Style', 'appilo' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_control(
			'button-text-color',
			[
				'label' => esc_html__( 'Button Color', 'appilo' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ft-btn-2 a span' => 'color: {{VALUE}}',
					'{{WRAPPER}} .ft-service-innerbox-2 .ft-service-text .ft-btn-2 a:hover i' => 'color: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render button widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $allowed_tags = wp_kses_allowed_html('post');
		
        $paged = get_query_var('paged');
		$paged = fastrans_set($_REQUEST, 'paged') ? esc_attr($_REQUEST['paged']) : $paged;

		$this->add_render_attribute( 'wrapper', 'class', 'templatepath-fastrans' );
		$args = array(
			'post_type'      => 'service',
			'posts_per_page' => fastrans_set( $settings, 'query_number' ),
			'orderby'        => fastrans_set( $settings, 'query_orderby' ),
			'order'          => fastrans_set( $settings, 'query_order' ),
			'paged'         => $paged
		);
		
		if( fastrans_set( $settings, 'query_category' ) ) $args['service_cat'] = fastrans_set( $settings, 'query_category' );
		$query = new \WP_Query( $args );

		if ( $query->have_posts() ) 
		{ ?>
		
        <!-- Start of Service page section
        ============================================= -->
        <section id="ft-service-page" class="ft-service-page-section page-padding">
            <div class="container">
                <?php if( $settings['subtitle'] || $settings['title'] ){ ?>
                <div class="ft-section-title-2 headline pera-content text-center">
                    <?php if( $settings['subtitle'] ){ ?><span class="sub-title"><?php echo wp_kses($settings['subtitle'], true)?></span><?php } ?>
                    <?php if( $settings['title'] ){ ?><h2><?php echo wp_kses($settings['title'], true)?></h2><?php } ?>
                </div>
                <?php } ?>
                <div class="ft-service-page-items">
                    <div class="row">
                        <?php $i = 1; while ( $query->have_posts() ) : $query->the_post(); ?>
                        <div class="col-lg-3 col-md-6">
                            <div class="ft-service-innerbox-2 position-relative">
                                <?php if(has_post_thumbnail()){ ?>
                                <div class="ft-service-img text-center">
                                    <?php the_post_thumbnail('fastrans_370x290'); ?>
                                </div>
                                <?php } ?>
                                <div class="ft-service-text position-relative headline">
                                    <div class="ft-service-icon position-absolute d-flex align-items-center justify-content-center">
                                        <i class="<?php echo wp_kses(str_replace( "icon ",  "",  get_post_meta(get_the_id(), 'service_icon', true )), $allowed_tags); ?>"></i>
                                    </div>
                                    <h3><a href="<?php echo esc_url(get_post_meta( get_the_id(), 'service_url', true ));?>"><?php the_title(); ?></a></h3>
                                    <div class="ft-btn-2">
                                        <a href="<?php echo esc_url(get_post_meta( get_the_id(), 'service_url', true ));?>">
                                            <i class="icon-first flaticon-right-arrow"></i>
                                            <span><?php esc_html_e('Read More', 'fastrans'); ?></span>
                                        </a>
                                    </div>
                                </div>
                                <div class="ft-service-serial position-absolute">
                                    <?php $i = sprintf('%2d', $i); echo $i; ?>
                                </div>
                            </div>
                        </div>
                        <?php $i++; endwhile; ?>
                    </div>
                    <?php if($settings['show_pagination']){ ?>
                    <div class="ft-pagination-item text-center ul-li">
                        <?php fastrans_the_pagination2(array('total'=>$query->max_num_pages, 'next_text' => '<i class="fa fa-angle-double-right"></i> ', 'prev_text' => '<i class="fa fa-angle-double-left"></i>')); ?>
                    </div>
                    <?php } ?>
                </div>
            </div>
        </section>
    	<!-- End of Service page section
        ============================================= -->
                    
        <?php }
		wp_reset_postdata();
	}
}