<?php
class HappyForms_Integration_Payments {

	private static $instance;

	private $frontend_styles = false;

	public $transaction_id_meta = 'transaction_id';
	public $transaction_details_meta = 'transaction_details';

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		self::$instance->hook();

		return self::$instance;
	}

	public function __construct() {
		$this->label = __( 'PayPal', 'happyforms' );
	}

	public function hook() {
		require_once( happyforms_get_integrations_folder() . '/services/payments/helper-payments.php' );
		require_once( happyforms_get_integrations_folder() . '/services/payments/class-part-payments.php' );

		require_once( happyforms_get_integrations_folder() . '/services/payments/class-coupon-controller.php' );
		require_once( happyforms_get_integrations_folder() . '/services/payments/class-coupon-admin.php' );

		add_filter( 'happyforms_message_part_value', array( $this, 'message_part_value' ), 10, 4 );
		add_filter( 'happyforms_email_part_value', array( $this, 'email_part_value' ), 10, 5 );
		add_filter( 'happyforms_stringify_part_value', array( $this, 'stringify_part_value' ), 10, 3 );
		add_filter( 'happyforms_default_validation_messages', array( $this, 'add_payment_validation_messages' ), 20 );
		add_filter( 'happyforms_messages_fields', array( $this, 'meta_messages_fields' ) );
		add_filter( 'happyforms_messages_controls', array( $this, 'get_message_controls' ) );
		add_action( 'happyforms_payment_the_transaction_details', array( $this, 'do_paypal_transaction_details' ), 10, 2 );
		add_action( 'happyforms_payment_the_transaction_details', array( $this, 'do_stripe_transaction_details' ), 10, 2 );
		add_filter( 'happyforms_get_csv_value', array( $this, 'get_csv_value' ), 10, 3 );
		add_filter( 'happyforms_get_form_data', array( $this, 'filter_preview_only_parts' ) );

		$payment_details_parts = $this->get_details_supported_parts();

		foreach ( $payment_details_parts as $part ) {
			add_filter( "happyforms_part_customize_fields_{$part}", array( $this, 'add_part_fields' ) );
			add_action( "happyforms_part_customize_{$part}_before_advanced_options", array( $this, 'add_part_controls' ) );
		}

		add_filter( 'happyforms_submission_is_pending', array( $this, 'submission_is_pending' ), 10, 3 );
		add_filter( 'happyforms_pending_submission_succeeded', array( $this, 'pending_submission_succeeded' ), 10, 2 );
	}

	public function append_response_transaction( $response_id, $gateway, $status = 'pending', $transaction_id = '', $transaction_details = array() ) {
		$transaction_details = wp_parse_args( $transaction_details, array(
			'gateway' => $gateway,
			'status' => $status,
			'transaction_id' => $transaction_id,
		) );
		$transaction_id = "{$gateway}:{$transaction_id}";

		happyforms_update_meta( $response_id, $this->transaction_id_meta, $transaction_id );
		happyforms_update_meta( $response_id, $this->transaction_details_meta, $transaction_details );
	}

	public function get_response_transaction( $response_id ) {
		$transaction_details = happyforms_get_meta( $response_id, $this->transaction_details_meta, true );

		return $transaction_details;
	}

	public function stringify_part_value( $value, $part, $form ) {
		if ( 'payments' !== $part['type'] ) {
			return $value;
		}

		if ( empty( $value['payment_method'] ) ) {
			$value = __( '', 'happyforms' );
			return $value;
		}

		$currencies = happyforms_payment_get_currencies();
		$currency = $currencies[$value['currency']];
		$currency_format = $currency['format'];
		$currency_symbol = $currency['symbol'];
		$price = floatval( $value['price'] );
		$decimals = 'float' === $currency_format ? 2 : 0;
		$gateway = $value['payment_method'];

		$service = happyforms_get_integrations()->get_service( $gateway );
		$gateway_label = $service->label;
		$price_value = number_format( $price, $decimals );
		$paid_through = "&rarr;";
		$string = "{$currency_symbol}{$price_value} {$paid_through} {$gateway_label}";

		$string = apply_filters( 'happyforms_get_currency_value', $string, $value, $value['currency'] );

		return $string;
	}

	public function message_part_value( $value, $original_value, $part, $context ) {
		if ( 'payments' !== $part['type'] ) {
			return $value;
		}

		if ( 'N/A' === $value ) {
			$value = '';
			
			return $value;
		}

		$value = htmlspecialchars_decode( $value );

		$details = '';

		if ( 'admin-column' === $context || 'admin-edit' === $context ) {
			global $post;

			$details = happyforms_payment_get_formatted_details( $post->ID, $context );
		} else if ( 'csv' === $context ) {
			global $happyforms_submission;

			$details = happyforms_payment_get_formatted_details( $happyforms_submission['ID'], $context );
		} else {
			return $value;
		}

		if ( '' === $details ) {
			$value = '';
			
			return $value;
		}

		$value = $value . $details;

		return $value;
	}

	public function email_part_value( $value, $message, $part, $form, $context ) {
		if ( 'payments' !== $part['type'] ) {
			return $value;
		}

		$response_id = $message['ID'];
		$details = happyforms_payment_get_formatted_details( $response_id, $context );

		if ( $details ) {
			$value = $value . $details;
		} else {
			$value = '';
		}

		return $value;
	}

	public function add_payment_validation_messages( $messages ) {
		$validation_messages = wp_list_pluck( $this->get_messages_fields(), 'default' );
		$messages = array_merge( $messages, $validation_messages );

		return $messages;
	}

	public function add_part_fields( $fields ) {
		$fields['add_to_payment_details'] = array(
			'default' => 0,
			'sanitize' => 'happyforms_sanitize_checkbox'
		);

		return $fields;
	}

	public function get_details_supported_parts() {
		return array(
			'single_line_text',
			'email',
			'website_url',
			'radio',
			'checkbox',
			'select',
			'number',
			'phone',
			'date',
			'title',
			'address',
		);
	}

	public function add_part_controls() {
		require( happyforms_get_integrations_folder() . '/services/payments/templates/partial-part-controls.php' );
	}

	public function get_transaction_description( $form, $submission ) {
		$description = array();

		foreach ( $form['parts'] as $part ) {
			if ( ! isset( $part['add_to_payment_details'] ) || 0 === (int) $part['add_to_payment_details'] ) {
				continue;
			}

			$part_id = $part['id'];
			$part_label = happyforms_get_part_label( $part );
			$description[] = "{$part_label}: {$submission[$part_id]}";
		}

		$description = implode( ', ', $description );

		return $description;
	}

	public function get_messages_fields() {
		$fields = array(
			'amount_too_low' => array(
				'default' => __( "This price isn't high enough.", 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'payment_completed' => array(
				'default' => __( 'Thank you! Your payment was successful.', 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'payment_failed' => array(
				'default' => __( 'Payment failed.', 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'payment_cancelled' => array(
				'default' => __( 'Payment canceled.', 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'payment_method_choice_label' => array(
				'default' => __( 'Choose a payment method', 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'user_price_label' => array(
				'default' => __( "Name a price", 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
		);

		return $fields;
	}

	public function meta_messages_fields( $fields ) {
		$fields = array_merge( $fields, $this->get_messages_fields() );

		return $fields;
	}

	public function get_message_controls( $controls ) {
		$message_controls = array(
			4340 => array(
				'type' => 'text',
				'label' => __( 'Price is too low', 'happyforms' ),
				'field' => 'amount_too_low',
			),
			200 => array(
				'type' => 'text',
				'label' => __( 'Payment completed', 'happyforms' ),
				'field' => 'payment_completed',
			),
			201 => array(
				'type' => 'text',
				'label' => __( 'Payment failed', 'happyforms' ),
				'field' => 'payment_failed',
			),
			202 => array(
				'type' => 'text',
				'label' => __( 'Payment cancelled', 'happyforms' ),
				'field' => 'payment_cancelled',
			),
			6135 => array(
				'type' => 'text',
				'label' => __( 'Payment method', 'happyforms' ),
				'field' => 'payment_method_choice_label',
			),
			6136 => array(
				'type' => 'text',
				'label' => __( 'Pay what you want', 'happyforms' ),
				'field' => 'user_price_label',
			),
		);

		$controls = happyforms_safe_array_merge( $controls, $message_controls );

		return $controls;
	}

	public function do_paypal_transaction_details( $details, $context ) {
		$gateway = $details['gateway'];
		$status = $details['status'];
		$transaction_id = $details['transaction_id'];

		if ( 'paypal' !== $gateway ) {
			return;
		}

		if ( 'confirmed' === $status || 'completed' === $status ) : ?>

			<?php
			$detail_url = 'https://www.paypal.com/activity/payment';

			if ( isset( $details['sandbox'] ) && happyforms_is_truthy( $details['sandbox'] ) ) {
				$detail_url = 'https://www.sandbox.paypal.com/activity/payment';
			}

			$detail_url = "{$detail_url}/$transaction_id";
			?>

			<?php if ( 'admin-column' === $context || 'admin-edit' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, <a href="<?php echo $detail_url; ?>" target="_blank"><?php _e( 'see details', 'happyforms' ); ?><?php happyforms_the_external_link_icon(); ?></a>)
			<?php elseif ( 'admin-email' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, <a href="<?php echo $detail_url; ?>" target="_blank"><?php _e( 'see details', 'happyforms' ); ?></a>)
			<?php elseif ( 'user-email' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>)
			<?php elseif ( 'csv' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, <?php echo $detail_url; ?>)
			<?php endif; ?>

		<?php else: ?>

			(<?php _e( 'canceled', 'happyforms' ); ?>)
		
		<?php endif;
	}

	public function do_stripe_transaction_details( $details, $context ) {
		$gateway = $details['gateway'];
		$status = $details['status'];
		
		if ( 'stripe' !== $gateway ) {
			return;
		}
		
		if ( 'confirmed' === $status || 'completed' === $status ) : ?>
			
			<?php $transaction_id = $details['transaction_id']; ?>

			<?php if ( 'admin-column' === $context || 'admin-edit' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, <a href="https://dashboard.stripe.com/payments/<?php echo $transaction_id; ?>" target="_blank"><?php _e( 'see details', 'happyforms' ); ?><?php happyforms_the_external_link_icon(); ?></a>)
			<?php elseif ( 'admin-email' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, <a href="https://dashboard.stripe.com/payments/<?php echo $transaction_id; ?>" target="_blank"><?php _e( 'see details', 'happyforms' ); ?></a>)
			<?php elseif ( 'user-email' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>)
			<?php elseif ( 'csv' === $context ) : ?>
				(<?php _e( 'completed', 'happyforms' ); ?>, https://dashboard.stripe.com/payments/<?php echo $transaction_id; ?>)
			<?php endif; ?>
		
		<?php else: ?>
			
			(<?php _e( 'canceled', 'happyforms' ); ?>)
		
		<?php endif;
	}

	public function get_csv_value( $value, $message, $part ) {
		if ( 'payments' !== $part['type'] ) {
			return $value;
		}

		$value = str_replace( '&rarr;', __( 'through', 'happyforms' ), $value );

		return $value;
	}

	public function submission_is_pending( $is_pending, $request, $form ) {
		$form = happyforms_get_conditional_controller()->get( $form, $request );
		$part = happyforms_get_form_controller()->get_first_part_by_type( $form, 'payments' );

		if ( ! $part ) {
			return false;
		}

		if ( 1 == $part['required'] ) {
			return true;
		}

		$part_class = happyforms_get_part_library()->get_part( $part['type'] );
		$part_name = happyforms_get_part_name( $part, $form );
		$sanitized_value = $part_class->sanitize_value( $part, $form, $request );
		$validated_value = $part_class->validate_value( $sanitized_value, $part, $form );

		if ( is_wp_error( $validated_value ) ) {
			return false;
		}

		if ( ! empty( $validated_value['payment_method'] ) ) {
			if ( 'stripe' === $validated_value['payment_method'] &&
				isset( $validated_value['filled'] ) &&
				'' !== $validated_value['filled'] ) {

				return true;
			}

			if ( 'paypal' === $validated_value['payment_method'] ) {
				return true;
			}
		}

		return $is_pending;
	}

	public function pending_submission_succeeded( $succeeded, $submission_id ) {
		$form_id = happyforms_get_meta( $submission_id, 'form_id', true );
		$form_controller = happyforms_get_form_controller();
		$message_controller = happyforms_get_message_controller();
		$form = $form_controller->get( $form_id );
		$submission = $message_controller->get( $submission_id );
		
		if ( $message_controller->submission_is_pending( $submission['request'], $form ) ) {
			$part = $form_controller->get_first_part_by_type( $form, 'payments' );

			if ( 1 == $part['required'] ) {
				$transaction_details = $this->get_response_transaction( $submission_id );

				if ( $transaction_details ) {
					$succeeded &= ( 'confirmed' === $transaction_details['status'] );
				}
			}
		}

		return $succeeded;
	}

	public function filter_preview_only_parts( $form ) {
		if ( happyforms_is_preview() || happyforms_is_form_edit_screen() || is_admin() ) {
			return $form;
		}

		if ( ! happyforms_has_active_payment( array(), $form ) ) {
			$parts = array();

			foreach ( $form['parts'] as $part ) {
				if ( 'payments' !== $part['type'] ) {
					$parts[] = $part;
				}
			}

			$form['parts'] = $parts;
		}

		return $form;
	}

}

if ( ! function_exists( 'happyforms_get_payments_integration' ) ):

function happyforms_get_payments_integration() {
	return HappyForms_Integration_Payments::instance();
}

endif;

happyforms_get_payments_integration();
